/*
 * @(#)OMClassFigure.java  2.0  2006-01-16
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.oo.diagram;

import ch.hslu.cm.*;
import ch.hslu.cm.simulation.*;
import ch.hslu.cm.oo.*;
import ch.hslu.cm.oo.objectmodel.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.io.IOException;
import java.util.*;
import javax.swing.*;
import org.jhotdraw.app.action.ActionUtil;
import static org.jhotdraw.draw.AttributeKeys.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.connector.ChopRectangleConnector;
import org.jhotdraw.draw.connector.Connector;
import org.jhotdraw.draw.connector.StickyRectangleConnector;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.layouter.VerticalLayouter;
import org.jhotdraw.geom.*;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.*;

/**
 * OMClassFigure visually represents a {@link OMClass}.
 * 
 * 
 * @author Werner Randelshofer
 * @version 2.0 2006-01-16 Rewritten to support double precision coordinates.
 * <br>1.0 1. Dezember 2003  Created.
 */
public class OMClassFigure
        extends GraphicalCompositeFigure implements DiagramFigure, OMClassListener {

    private OMClass model;

    /**
     * This adapter is used, to connect a TextFigure with the name of
     * the OMClass model.
     */
    private static class NameAdapter extends FigureAdapter {

        private OMClassFigure target;

        public NameAdapter(OMClassFigure target) {
            this.target = target;
        }

        @Override
        public void attributeChanged(FigureEvent e) {
            if (e.getAttribute().equals(TEXT)) {
                target.model.setName((String) e.getNewValue());
            }
        }
    }

    /**
     * This adapter is used, to connect a TextFigure for an attribute with
     * the OMClass model.
     */
    private class AttributeAdapter extends FigureAdapter {

        private int index;

        public AttributeAdapter(int index) {
            this.index = index;
        }

        @Override
        public void attributeChanged(FigureEvent e) {
            if (e.getAttribute().equals(TEXT)) {
                String text = (String) e.getNewValue();
                if (text == null || text.length() == 0) {
                    model.removeAttribute(index);
                } else {
                    model.setAttribute(index, text);
                }
            }
        }
    }

    /**
     * This adapter is used, to connect a TextFigure for an operation with
     * the OMClass model.
     */
    private class OperationAdapter extends FigureAdapter {

        private int index;

        public OperationAdapter(int index) {
            this.index = index;
        }

        @Override
        public void attributeChanged(FigureEvent e) {
            if (e.getAttribute().equals(TEXT)) {
                String text = (String) e.getNewValue();
                if (text == null || text.length() == 0) {
                    model.removeOperation(index);
                } else {
                    model.setOperationName(index, text);
                }
            }
        }
    }

    /**
     * Gets a connector for this figure at the given location.
     * A figure can have different connectors at different locations.
     */
    @Override
    public Connector findConnector(Point2D.Double p, ConnectionFigure prototype) {
        return new StickyRectangleConnector(this, p);
    }

    private ListFigure getNameCompartment() {
        return (ListFigure) getChild(0);
    }

    private ListFigure getAttributeCompartment() {
        return (ListFigure) getChild(2);
    }

    private ListFigure getOperationCompartment() {
        return (ListFigure) getChild(4);
    }

    /** Creates a new instance. */
    public OMClassFigure() {
        // super(new ShadowRectangleFigure());
        super(new RectangleFigure());

        setLayouter(new VerticalLayouter());

        RectangleFigure nameCompartmentPF = new RectangleFigure();
        STROKE_COLOR.set(nameCompartmentPF, null);
        nameCompartmentPF.setAttributeEnabled(STROKE_COLOR, false);
        FILL_COLOR.set(nameCompartmentPF, null);
        nameCompartmentPF.setAttributeEnabled(FILL_COLOR, false);
        ListFigure nameCompartment = new ListFigure(nameCompartmentPF);
        ListFigure attributeCompartment = new ListFigure();
        ListFigure operationCompartment = new ListFigure();
        SeparatorLineFigure separator1 = new SeparatorLineFigure();
        SeparatorLineFigure separator2 = new SeparatorLineFigure();

        applyAttributes(getPresentationFigure());

        add(nameCompartment);
        add(separator1);
        add(attributeCompartment);
        add(separator2);
        add(operationCompartment);

        Insets2D.Double insets = new Insets2D.Double(4, 8, 4, 8);
        LAYOUT_INSETS.set(nameCompartment, insets);
        LAYOUT_INSETS.set(attributeCompartment, insets);
        LAYOUT_INSETS.set(operationCompartment, insets);

        TextFigure nameFigure;
        nameCompartment.add(nameFigure = new TextFigure());
        nameFigure.addFigureListener(new NameAdapter(this));
        FONT_BOLD.set(nameFigure, true);

        applyAttributes(this);
        setModel(createSimulatedClass());

        setAttributeEnabled(STROKE_DASHES, false);
    }

    protected OMClass createSimulatedClass() {
        return new OMClass();
    }

    private void updateAttributeCompartment() {
        getAttributeCompartment().removeAllChildren();
        for (int i = 0; i < model.getAttributeCount(); i++) {
            TextFigure f = new TextFigure(model.getAttribute(i).getName());
            f.addFigureListener(new AttributeAdapter(i));
            f.set(CompositeFigure.LAYOUT_INSETS, new Insets2D.Double(1, 0, 3, 0));
            applyAttributes(f);
            getAttributeCompartment().add(f);
        }
    }

    private void updateOperationCompartment() {
        getOperationCompartment().removeAllChildren();
        for (int i = 0; i < model.getOperationCount(); i++) {
            TextFigure f = new TextFigure(model.getOperation(i).getName());
            f.set(FONT_ITALIC, model.getOperation(i).isAbstract());
            f.addFigureListener(new OperationAdapter(i));
            f.set(CompositeFigure.LAYOUT_INSETS, new Insets2D.Double(1, 0, 3, 0));
            applyAttributes(f);
            getOperationCompartment().add(f);
        }
    }

    public void setModel(OMClass m) {
        willChange();
        if (model != null) {
            model.removeSimulatedClassListener(this);
        }
        model = m;
        if (model != null) {
            model.addSimulatedClassListener(this);
            ((TextFigure) getNameCompartment().getChild(0)).setText(model.getName());
            ((TextFigure) getNameCompartment().getChild(0)).set(FONT_ITALIC, model.isAbstract());

            updateAttributeCompartment();
            updateOperationCompartment();

        }
        layout();
        changed();
    }

    @Override
    public OMClass getModel() {
        return model;
    }

    private Diagram getDiagram() {
        return (Diagram) getDrawing();
    }

    private Simulation getSimulation() {
        return getDiagram().getSimulation();
    }

    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && getModel() != null) {
            getSimulation().add(getModel());
        }
    }

    @Override
    public void removeNotify(Drawing drawing) {
        if (getDrawing() != null && getModel() != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }

    private void applyAttributes(Figure f) {
        Map<AttributeKey, Object> attr = ((AbstractAttributedFigure) getPresentationFigure()).getAttributes();
        for (Map.Entry<AttributeKey, Object> entry : attr.entrySet()) {
            f.set(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public Collection<Action> getActions(Point2D.Double p) {
        ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.oo.Labels");
        LinkedList<Action> actions = new LinkedList<Action>();
        Action action;

        Figure compartment = findCompartment(p);
        Figure item;

        action = new AbstractAction(labels.getString("action.toggleAbstractClass.text")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setAbstract(!model.isAbstract());
            }
        };
        actions.add(action);

        action = new AbstractAction(labels.getString("action.add.text")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.oo.Labels");
                model.addAttribute(labels.getString("model.attribute.name.text"));
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("menu.attribute.text"));
        actions.add(action);

        if (compartment == getAttributeCompartment()) {
            final int index = getAttributeCompartment().findChildIndex(p);
            if (index != -1) {
                action = new AbstractAction(labels.getFormatted("action.removeX.text", getModel().getAttribute(index).getAttributeName())) {

                    @Override
                    public void actionPerformed(ActionEvent event) {
                        model.removeAttribute(index);
                    }
                };
                action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("menu.attribute.text"));
                actions.add(action);
            }
        }

        action = new AbstractAction(labels.getString("action.add.text")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.oo.Labels");
                model.addOperation(labels.getString("model.operation.name.text"));
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("menu.operation.text"));
        actions.add(action);

        if (compartment == getOperationCompartment()) {
            final int index = getOperationCompartment().findChildIndex(p);
            if (index != -1) {
                action = new AbstractAction(labels.getFormatted("action.removeX.text", getModel().getOperation(index).getName())) {

                    @Override
                    public void actionPerformed(ActionEvent event) {
                        model.removeOperation(index);
                    }
                };
                action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("menu.operation.text"));
                actions.add(action);
                action = new AbstractAction(labels.getFormatted("action.editMethodX.text", getModel().getOperation(index).getName())) {

                    @Override
                    public void actionPerformed(ActionEvent event) {
                        OMMethodFigure mf = new OMMethodFigure();
                        mf.setModel(OMClassFigure.this, index);
                        Rectangle2D.Double b = getBounds();
                        mf.setBounds(
                                new Point2D.Double(b.x + b.width + 10, b.y),
                                new Point2D.Double(b.x + b.width * 2 + 10, b.y + b.height));
                        getDrawing().add(mf);
                        OMCommentLinkFigure lc = new OMCommentLinkFigure();
                        //lc.addNode(new BezierPath.Node());
                        //lc.addNode(new BezierPath.Node());
                        lc.setStartConnector(new ChopRectangleConnector(OMClassFigure.this));
                        lc.setEndConnector(new ChopRectangleConnector(mf));
                        STROKE_DASHES.set(lc, new double[]{4d, 4d});
                        getDrawing().add(lc);
                        lc.updateConnection();
                    }
                };
                action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("menu.operation.text"));
                actions.add(action);
        action = new AbstractAction(labels.getFormatted("action.toggleAbstractOperationX.text", getModel().getOperation(index).getName())) {

            @Override
            public void actionPerformed(ActionEvent event) {
                OMOperation op=model.getOperation(index);
                op.setAbstract(!op.isAbstract());
                willChange();
                updateOperationCompartment();
                changed();
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("menu.operation.text"));
        actions.add(action);
            }
        }

        return actions;
    }

    public Figure findCompartment(Point2D.Double p) {
        if (getNameCompartment().contains(p)) {
            return getNameCompartment();
        }
        if (getAttributeCompartment().contains(p)) {
            return getAttributeCompartment();
        }
        if (getOperationCompartment().contains(p)) {
            return getOperationCompartment();
        }
        return null;
    }
    /*
    public Collection createHandles() {
    LinkedList handles = new LinkedList();
    NullHandle.addLeadHandles(this, handles);
    return handles;
    }*/

    @Override
    public OMClassFigure clone() {
        OMClassFigure that = (OMClassFigure) super.clone();
        that.setModel((OMClass) this.model.clone());
        that.getNameCompartment().getChild(0).addFigureListener(new NameAdapter(that));
        return that;
    }

    public void remap(HashMap<Figure, Figure> oldToNew) {
        /*
        nameCompartment = (ListFigure) oldToNew.get(nameCompartment);
        attributeCompartment = (ListFigure) oldToNew.get(attributeCompartment);
        operationCompartment = (ListFigure) oldToNew.get(operationCompartment);
        nameCompartment.getChild(0).addFigureListener(new NameAdapter(this));
         */
    }

    /**
     * Delegate capabilities for storing and retrieving attributes to a
     * CompositeFigure if the encapsulated presentation figure. If no
     * presentation figure is found then the superclass' addAttribute()
     * will be invoked (which currently does not set an attribute).
     *
     *
     * @param name	name of the attribute
     * @param value	value associated with this attribute
     */
    @Override
    public void set(AttributeKey name, Object value) {
        if (name.equals(FONT_BOLD)
                || name.equals(FONT_ITALIC)
                || name.equals(FONT_UNDERLINE)) {
        } else {
            willChange();
            if (getPresentationFigure() != null) {
                //if (name.equals(FILL_COLOR)) {
                //    getPresentationFigure().set(name, ClassDiagram.CLASS_ATTRIBUTE_COMPARTMENT_COLOR);
                //} else {
                getPresentationFigure().set(name, value);
                //}
            }
            for (Iterator i = getChildren().iterator(); i.hasNext();) {
                Figure child = (Figure) i.next();

                child.set(name, value);
            }

            Object oldValue = attributes.put(name, value);
            fireAttributeChanged(name, oldValue, value);
            changed();
        }
    }

    @Override
    public void attributeAdded(OMClassEvent event) {
        willChange();
        updateAttributeCompartment();
        //    layout();
        changed();
    }

    @Override
    public void attributeChanged(OMClassEvent event) {
        willChange();
        ((TextFigure) getAttributeCompartment().getChild(event.getIndex())).setText(model.getAttribute(event.getIndex()).getName());
        //      layout();
        changed();
    }

    @Override
    public void attributeRemoved(OMClassEvent event) {
        willChange();
        updateAttributeCompartment();
//        layout();
        changed();
    }

    @Override
    public void operationAdded(OMClassEvent event) {
        willChange();
        updateOperationCompartment();
        //layout();
        changed();
    }

    @Override
    public void operationChanged(OMClassEvent event) {
        willChange();
        ((TextFigure) getOperationCompartment().getChild(event.getIndex())).setText(model.getOperation(event.getIndex()).getName());
        //layout();
        changed();
    }

    @Override
    public void operationRemoved(OMClassEvent event) {
        willChange();
        updateOperationCompartment();
        //layout();
        changed();
    }

    @Override
    public void nameChanged(OMClassEvent event) {
        willChange();
        ((TextFigure) getNameCompartment().getChild(0)).setText(model.getName());
        //layout();
        changed();
    }

    @Override
    public void abstractChanged(OMClassEvent event) {
        willChange();
        ((TextFigure) getNameCompartment().getChild(0)).set(FONT_ITALIC, model.isAbstract());
        //layout();
        changed();
    }

    public int getConnectionCount() {
        return getModel().getRelationships().size();
    }

    public int getConnectionIndex(DiagramFigure f) {
        System.out.println("ClassFigure  " + getModel() + ".getConnectionIndex(" + f.getModel() + ")");
        System.out.println("    " + getModel().getRelationships(f.getModel().getSimulatedConcept()).indexOf(f.getModel()));
        return getModel().getRelationships().indexOf(f.getModel());
    }

    @Override
    public void generalizationChanged(OMClassEvent event) {
    }

    protected void drawConnectors(Graphics2D g) {
        Rectangle2D.Double r = getBounds();
        Geom.grow(r, 1d, 1d);
        g.setColor(ClassDiagram.CLASS_CONNECTOR_COLOR);
        g.setStroke(new BasicStroke());
        g.draw(r);
    }

    @Override
    public Connector findCompatibleConnector(Connector c, boolean isStart) {
        if (c instanceof StickyRectangleConnector) {
            Point2D.Double p = c.getAnchor();
            return new StickyRectangleConnector(this, p);
        } else {
            return new ChopRectangleConnector(this);
        }
    }

    @Override
    public void read(DOMInput in) throws IOException {
        double x = in.getAttribute("x", 0d);
        double y = in.getAttribute("y", 0d);
        double w = in.getAttribute("w", 0d);
        double h = in.getAttribute("h", 0d);
        setBounds(new Point2D.Double(x, y), new Point2D.Double(x + w, y + h));
        readAttributes(in);
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((OMClass) in.readObject(0));
        in.closeElement();
    }

    @Override
    public void write(DOMOutput out) throws IOException {
        Rectangle2D.Double r = getBounds();
        out.addAttribute("x", r.x);
        out.addAttribute("y", r.y);
        writeAttributes(out);
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
    }

    @Override
    public int getLayer() {
        return ClassDiagram.CLASS_LAYER;
    }
}
